//	TorusGamesHelpChoiceController.m
//
//	© 2021 by Jeff Weeks
//	See TermsOfUse.txt

#import "TorusGamesHelpChoiceController.h"
#import "GeometryGamesWebViewController.h"
#import "GeometryGamesUtilities-iOS.h"
#import "GeometryGamesUtilities-Mac-iOS.h"
#import "GeometryGamesLocalization.h"


struct
{
	Char16	*itsName,
			*itsDirectoryName,
			*itsFileName;
	bool	itsFileIsLocalized;
	Char16	*itsImageName;
} gHelpInfo[2][5] =
{
	{
		{u"Wraparound Universe",	u"Help - legacy format",	u"WraparoundUniverse-%@.html",	true,	u"Table Images/Help/Wraparound Universe"	},
		{u"How to Play - 2D",		u"Help - legacy format",	u"HowToPlay-2D-%@.html",		true,	u"Table Images (shared)/Help/HowTo"			},
		{u"How to Play - 3D",		u"Help - legacy format",	u"HowToPlay-3D-%@.html",		true,	u"Table Images (shared)/Help/HowTo"			},
		{u"Practice Board",			NULL,						NULL,							false,	u"Table Images/Help/Practice Board"			},
		{u"Contact",				u"Help - legacy format",	u"Contact-%@.html",				true,	u"Table Images (shared)/Help/Contact"		}
	},
	{
		{u"Translators",			u"Thanks",	u"Translators.html",			false,	u"Table Images (shared)/Thanks/Translators"	},
		{u"Math for America",		u"Thanks",	u"MathForAmerica.html",			false,	u"Table Images/Thanks/MfA"					},
		{u"NSF",					u"Thanks",	u"NSF.html",					false,	u"Table Images (shared)/Thanks/NSF"			},
		{u"UNUSED",					NULL,		NULL,							false,	NULL										},
		{u"UNUSED",					NULL,		NULL,							false,	NULL										}
	}
};


@implementation TorusGamesHelpChoiceController
{
	NSObject<TorusGamesHelpChoiceDelegate>  * __weak	itsDelegate;
	
	//	If we change the game, we'd like to update the table
	//	so that the Practice Board does or doesn't get a check mark,
	//	according to whether it's selected or not.  In principle
	//	we may call [itsDelegate practiceBoardIsSelected] to find out
	//	whether it's selected.  In practice it's trickier than that,
	//	because a call to [itsDelegate requestGame:] doesn't change
	//	the game immediately, but just starts an animation running,
	//	at whose halfway point the game will change.
	//	So let's keep track of the selected game privately,
	//	so we can update our private value immediately.
	//
	//	Note:  An alternative approach would be to let the change-game
	//	animation send us a message when it reaches it halfway point,
	//	but that would complicate the structure of app as a whole.
	//	Better to restrict the kludge to this file only,
	//	and let the rest of the code stay simple.
	//
	bool			itsPracticeBoardIsSelected;	//	cached value -- see explanation in comment above

	UIBarButtonItem	*itsCloseButton;
}


- (id)initWithDelegate:(NSObject<TorusGamesHelpChoiceDelegate> *)aDelegate
{
	self = [super initWithStyle:UITableViewStyleGrouped];
	if (self != nil)
	{
		itsDelegate					= aDelegate;
		itsPracticeBoardIsSelected	= [aDelegate practiceBoardIsSelected];
		
		[self setTitle:GetLocalizedTextAsNSString(u"Help")];
		
		itsCloseButton = [[UIBarButtonItem alloc]
			initWithTitle:	GetLocalizedTextAsNSString(u"Close")
			style:			UIBarButtonItemStyleDone
			target:			aDelegate
			action:			@selector(dismissHelp)];
	}
	return self;
}

- (BOOL)prefersStatusBarHidden
{
	return [itsDelegate prefersStatusBarHidden];
}

- (void)viewWillAppear:(BOOL)animated
{
	[super viewWillAppear:animated];

	[self setPreferredContentSize:(CGSize){HELP_PICKER_WIDTH, HELP_PICKER_HEIGHT}];

	[self adaptNavBarForHorizontalSize:
		[[RootViewController(self) traitCollection] horizontalSizeClass]];
}


#pragma mark -
#pragma mark GeometryGamesPopover

- (void)adaptNavBarForHorizontalSize:(UIUserInterfaceSizeClass)aHorizontalSizeClass
{
	//	When the presenting view is horizontally compact,
	//	the Help panel needs a Close button, because otherwise
	//	the user would have no way to dismiss it.
	//
	//	When the presenting view is horizontally regular,
	//	touching the matte would dismiss the Help panel,
	//	but touches to the game itself get passed through,
	//	and the user might not think to try touching the matte.
	//	To avoid frustration, let's provide an explicit Close button
	//	in all cases.
	//
	[[self navigationItem] setRightBarButtonItem:itsCloseButton];
}


#pragma mark -
#pragma mark UITableViewDataSource

- (NSInteger)numberOfSectionsInTableView:(UITableView *)aTableView
{
	return 2;
}

- (NSString *)tableView:(UITableView *)aTableView titleForHeaderInSection:(NSInteger)aSection
{
	switch (aSection)
	{
		case 0:		return nil;
		case 1:		return GetLocalizedTextAsNSString(u"Thanks");
		default:	return @"internal error";
	}
}

- (NSInteger)tableView:(UITableView *)aTableView numberOfRowsInSection:(NSInteger)aSection
{
	switch (aSection)
	{
#ifdef SCIENCE_CENTER_VERSION
		case 0:		return 4;	//	Suppress Practice Board table cell, which is useful only on iPhone.
#else
		case 0:		return 5;
#endif
		case 1:		return 3;
		default:	return 0;
	}
}

- (UITableViewCell *)tableView:(UITableView *)aTableView cellForRowAtIndexPath:(NSIndexPath *)anIndexPath
{
	UITableViewCell	*theCell;
	UILabel			*theLabel;
	NSUInteger		theSection,
					theRow;

	theCell		= [[UITableViewCell alloc] initWithStyle:UITableViewCellStyleDefault reuseIdentifier:nil];
	theLabel	= [theCell textLabel];
	theSection	= [anIndexPath section];
	theRow		= [anIndexPath row];

#ifdef SCIENCE_CENTER_VERSION
	//	The Contact table cell is normally at row 4,
	//	but in the Science Center Version it moves to row 3,
	//	because Practice Board table cell is gone.
	if (theSection == 0 && theRow == 3)
		theRow = 4;
#endif

	[theLabel setText:
		GetLocalizedTextAsNSString(gHelpInfo[theSection][theRow].itsName)];

	[[theCell imageView] setImage:[UIImage imageNamed:
		GetNSStringFromZeroTerminatedString(gHelpInfo[theSection][theRow].itsImageName)]];
	
	if (gHelpInfo[theSection][theRow].itsFileName != nil)
	{
		[theCell setAccessoryType:UITableViewCellAccessoryDisclosureIndicator];
	}
	else
	{
		//	theCell represents the Practice Board.
		//	Add a checkmark iff the Practice Board is active.
		[theCell setAccessoryType:(itsPracticeBoardIsSelected ?
				UITableViewCellAccessoryCheckmark : UITableViewCellAccessoryNone)];
	}

	return theCell;
}


#pragma mark -
#pragma mark UITableViewDelegate

- (NSIndexPath *)tableView:(UITableView *)aTableView willSelectRowAtIndexPath:(NSIndexPath *)anIndexPath
{
	NSUInteger						theSection,
									theRow;
	NSString						*theDirectoryName,
									*theFileBaseName,
									*theLanguageSuffix,
									*theFileName;
	GeometryGamesWebViewController	*theWebViewController;

#ifdef SCIENCE_CENTER_VERSION
	[itsDelegate userIsStillPlaying];
#endif

	theSection			= [anIndexPath section];
	theRow				= [anIndexPath row];
#ifdef SCIENCE_CENTER_VERSION
	//	The Contact table cell is normally at row 4,
	//	but in the Science Center Version it moves to row 3,
	//	because Practice Board table cell is gone.
	if (theSection == 0 && theRow == 3)
		theRow = 4;
#endif
	theDirectoryName	= GetNSStringFromZeroTerminatedString(gHelpInfo[theSection][theRow].itsDirectoryName);
	theFileBaseName		= GetNSStringFromZeroTerminatedString(gHelpInfo[theSection][theRow].itsFileName		);

	if (theFileBaseName != nil)	//	request for HTML page
	{
		if (gHelpInfo[theSection][theRow].itsFileIsLocalized)
		{
			//	My legacy system used a strictly 2-letter language code.
			//	To allow for simplified and traditional Chinese,
			//	it used two exceptional codes
			//
			//		zs -> zh-Hans
			//		zt -> zh-Hant
			//
			if (IsCurrentLanguage(u"zs"))
				theLanguageSuffix = @"zh-Hans";
			else
			if (IsCurrentLanguage(u"zt"))
				theLanguageSuffix = @"zh-Hant";
			else
				theLanguageSuffix = GetNSStringFromZeroTerminatedString(GetCurrentLanguage());

			theFileName = [NSString stringWithFormat:theFileBaseName, theLanguageSuffix];
		}
		else
		{
			theFileName = theFileBaseName;
		}

		theWebViewController = [[GeometryGamesWebViewController alloc]
				initWithDirectory:		theDirectoryName
				page:					theFileName
				closeButton:			itsCloseButton
				showCloseButtonAlways:	YES
				hideStatusBar:			[self prefersStatusBarHidden]
			];
		[[self navigationController] pushViewController:theWebViewController animated:YES];
	}
	else
	{
		//	The user has pressed the "Practice Board" item.

		//	Replace the current game with the practice board.

		itsPracticeBoardIsSelected = true;
		[[self tableView] reloadData];	//	to add a checkmark to the Practice Board item

		//	In a horizontally compact layout, selectPracticeBoard will dismiss
		//	the Help panel so that the user may see the newly selected practice board,
		//	but in a horizontally regular layout it will leave the Help panel up.
		//
		//		Caution:  In a horizontally compact layout, selectPracticeBoard
		//		will cause this TorusGamesHelpChoiceController to be deallocated,
		//		which could leave "self" invalid (see discussion immediately below).
		//		In practice, dismissViewControllerAnimated:completion: doesn't
		//		deallocate the presented view controller immediately (even if
		//		its "animated" parameter is NO) and nothing bad happens,
		//		but it nevertheless seems like a bad idea to rely on a "self"
		//		object that could be deallocated at any moment.
		//
		[itsDelegate selectPracticeBoard];
	}
	
	//	The Clang documentation at
	//
	//		http://clang.llvm.org/docs/AutomaticReferenceCounting.html#self
	//
	//	says that
	//
	//		The self parameter variable of an Objective-C method
	//		is never actually retained by the implementation.
	//		It is undefined behavior, or at least dangerous,
	//		to cause an object to be deallocated during a message send
	//		to that object.
	//
	//	So no further references to self should appear here.

	return nil;
}

@end
